# 🤖 VYLO AI Integration Quick Start

## Get Your API Keys

### 1. Claude (Anthropic) - Code Generation
```bash
# Sign up: https://console.anthropic.com/
# Get API key from dashboard
ANTHROPIC_API_KEY=sk-ant-...
```

### 2. OpenAI (GPT-4 + DALL-E) - Content & Images
```bash
# Sign up: https://platform.openai.com/
# Get API key from dashboard
OPENAI_API_KEY=sk-...
```

### 3. Midjourney (Optional)
- More complex, requires Discord bot setup
- Alternative: Use DALL-E for now

---

## Install AI SDKs

```bash
cd C:\Users\Digit\source\repos\vylo
npm install @anthropic-ai/sdk openai ai
```

---

## Add Environment Variables

Create `.env.local` in project root:

```env
# Database
DATABASE_URL="file:./prisma/dev.db"

# AI APIs
ANTHROPIC_API_KEY=your-claude-key-here
OPENAI_API_KEY=your-openai-key-here

# App
NEXT_PUBLIC_APP_URL=http://localhost:3000
```

---

## Quick AI Integration Example

### File: `app/api/ai/generate-code/route.ts`

```typescript
import { NextRequest, NextResponse } from "next/server";
import Anthropic from "@anthropic-ai/sdk";

const anthropic = new Anthropic({
  apiKey: process.env.ANTHROPIC_API_KEY,
});

export async function POST(req: NextRequest) {
  const { prompt, language } = await req.json();

  const message = await anthropic.messages.create({
    model: "claude-sonnet-4-5-20250929",
    max_tokens: 4096,
    messages: [{
      role: "user",
      content: `Generate ${language} code for: ${prompt}`,
    }],
  });

  const code = message.content[0].text;

  return NextResponse.json({ code });
}
```

### File: `app/api/ai/generate-logo/route.ts`

```typescript
import { NextRequest, NextResponse } from "next/server";
import OpenAI from "openai";

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

export async function POST(req: NextRequest) {
  const { description } = await req.json();

  const response = await openai.images.generate({
    model: "dall-e-3",
    prompt: `Professional logo design: ${description}`,
    n: 1,
    size: "1024x1024",
    quality: "hd",
  });

  const imageUrl = response.data[0].url;

  return NextResponse.json({ imageUrl });
}
```

---

## Usage in VYLO

### Logo Generator Example

```typescript
// In your component
const generateLogo = async () => {
  const res = await fetch("/api/ai/generate-logo", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
      description: "Modern tech startup logo, minimalist, purple and cyan colors",
    }),
  });

  const { imageUrl } = await res.json();
  // Display the logo!
};
```

### Code Generator Example

```typescript
const generateCode = async () => {
  const res = await fetch("/api/ai/generate-code", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
      prompt: "Create a React login form with email and password",
      language: "typescript",
    }),
  });

  const { code } = await res.json();
  // Show the generated code!
};
```

---

## Next Steps

1. **Get API Keys** (5 min)
2. **Install SDKs** (1 min)
3. **Create .env.local** (1 min)
4. **Copy AI route examples** (5 min)
5. **Test in browser** (2 min)

**Total: ~15 minutes to have working AI!**

---

## Testing

```bash
# Test logo generation
curl -X POST http://localhost:3000/api/ai/generate-logo \
  -H "Content-Type: application/json" \
  -d '{"description":"Futuristic AI assistant logo"}'

# Test code generation
curl -X POST http://localhost:3000/api/ai/generate-code \
  -H "Content-Type: application/json" \
  -d '{"prompt":"Hello world function","language":"javascript"}'
```

---

## Cost Management

- **Claude**: ~$3 per 1M input tokens, $15 per 1M output
- **GPT-4**: ~$2.50 per 1M input, $10 per 1M output
- **DALL-E 3**: ~$0.04 per image (HD)

**For MVP testing: $20-50 should be plenty!**

---

## Production Tips

1. Add rate limiting
2. Cache AI responses
3. Queue system for long tasks
4. Error handling & retries
5. User credit system
6. Stream responses for better UX

---

**Ready to add AI power? Just follow this guide!** 🚀
