"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";

export default function UserMenu() {
  const router = useRouter();
  const [isOpen, setIsOpen] = useState(false);
  const [user, setUser] = useState<{ email: string; name?: string | null } | null>(null);

  useEffect(() => {
    // Fetch current user from API
    fetch("/api/auth/me")
      .then(res => res.json())
      .then(data => {
        if (data.user) {
          setUser(data.user);
        }
      })
      .catch(() => {
        // Ignore errors
      });
  }, []);

  const handleLogout = async () => {
    await fetch("/api/auth/logout", { method: "POST" });
    router.push("/login");
  };

  if (!user) {
    return (
      <div className="w-10 h-10 rounded-full bg-gradient-to-br from-vylo-secondary to-vylo-accent animate-pulse"></div>
    );
  }

  return (
    <div className="relative">
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="w-10 h-10 rounded-full bg-gradient-to-br from-vylo-secondary to-vylo-accent cursor-pointer hover:scale-110 transition-transform flex items-center justify-center font-bold text-white"
      >
        {user.name?.[0]?.toUpperCase() || user.email[0].toUpperCase()}
      </button>

      {isOpen && (
        <>
          <div
            className="fixed inset-0 z-10"
            onClick={() => setIsOpen(false)}
          ></div>
          <div className="absolute right-0 mt-2 w-64 vylo-card z-20">
            <div className="px-4 py-3 border-b border-vylo-accent/20">
              <p className="font-semibold">{user.name || "User"}</p>
              <p className="text-sm text-gray-400 truncate">{user.email}</p>
            </div>
            <div className="py-2">
              <button
                onClick={handleLogout}
                className="w-full px-4 py-2 text-left hover:bg-vylo-darker transition-colors text-red-400"
              >
                Sign out
              </button>
            </div>
          </div>
        </>
      )}
    </div>
  );
}
